package com.yd.user.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.user.api.service.ApiRelTenantProjectService;
import com.yd.user.feign.request.reltenantproject.ApiRelTenantProjectListAddRequest;
import com.yd.user.feign.request.reltenantproject.ApiRelTenantProjectListPageRequest;
import com.yd.user.feign.request.reltenantproject.ApiRelTenantProjectSelectPageRequest;
import com.yd.user.feign.response.reltenantproject.ApiRelTenantProjectListPageResponse;
import com.yd.user.feign.response.reltenantproject.ApiRelTenantProjectSelectPageResponse;
import com.yd.user.service.model.RelTenantProject;
import com.yd.user.service.service.IRelTenantProjectService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
public class ApiRelTenantProjectServiceImpl implements ApiRelTenantProjectService {

    @Autowired
    private IRelTenantProjectService iRelTenantProjectService;

    /**
     * 租户和项目关系分页查询
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiRelTenantProjectListPageResponse>> page(ApiRelTenantProjectListPageRequest request) {
        Page<ApiRelTenantProjectListPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiRelTenantProjectListPageResponse> iPage = iRelTenantProjectService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 添加租户和项目列表关系
     * @param request
     * @return
     */
    @Override
    public Result addProjectList(ApiRelTenantProjectListAddRequest request) {
        List<RelTenantProject> list = new ArrayList<>();
        if (CollectionUtils.isEmpty(request.getProjectBizIdList())) {
            //项目唯一标识(业务ID) 列表（选中列表）不为空
            list = request.getProjectBizIdList().stream().map(bizId -> {
                RelTenantProject relTenantProject = new RelTenantProject();
                relTenantProject.setTenantBizId(request.getTenantBizId());
                relTenantProject.setProjectBizId(bizId);
                return relTenantProject;
            }).collect(Collectors.toList());

            iRelTenantProjectService.saveOrUpdateBatch(list);
        }

        return Result.success();
    }

    /**
     * 删除租户和项目关系
     * @param id 租户和项目关系表主键id
     * @return
     */
    @Override
    public Result del(Long id) {
        RelTenantProject relTenantProject = iRelTenantProjectService.queryOne(id);
        if (Objects.isNull(relTenantProject)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        iRelTenantProjectService.removeById(relTenantProject.getId());
        return Result.success();
    }

    /**
     * 导入-选择项目分页查询
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiRelTenantProjectSelectPageResponse>> selectProjectPage(ApiRelTenantProjectSelectPageRequest request) {
        Page<ApiRelTenantProjectSelectPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiRelTenantProjectSelectPageResponse> iPage = iRelTenantProjectService.selectProjectPage(page, request);
        return Result.success(iPage);
    }

    /**
     * 添加单个租户和项目关系
     * @param tenantBizId
     * @param projectBizId
     * @return
     */
    @Override
    public Result addSingleRel(String tenantBizId, String projectBizId) {
        //新增关系
        RelTenantProject relTenantProject1 = new RelTenantProject();
        relTenantProject1.setTenantBizId(tenantBizId);
        relTenantProject1.setProjectBizId(projectBizId);
        iRelTenantProjectService.saveOrUpdate(relTenantProject1);
        return Result.success();
    }

    /**
     * 根据项目业务id批量删除租户和项目关系
     * @param projectBizId
     * @return
     */
    @Override
    public Result delRelByProjectBizId(String projectBizId) {
        iRelTenantProjectService.delRelByProjectBizId(projectBizId);
        return Result.success();
    }

    /**
     * 根据项目id删除关系（保留传入租户id列表的关系）
     * @param projectBizId
     * @return
     */
    @Override
    public Result delRelByProjectBizIdNeTenantBizIdList(String projectBizId, List<String> tenantBizIdList) {
        iRelTenantProjectService.delRelByProjectBizIdNeTenantBizIdList(projectBizId,tenantBizIdList);
        return Result.success();
    }

    /**
     * 根据租户业务id批量删除租户和项目关系
     * @param tenantBizId
     * @return
     */
    @Override
    public Result delRelByTenantBizId(String tenantBizId) {
        iRelTenantProjectService.delRelByTenantBizId(tenantBizId);
        return Result.success();
    }


}
