package com.yd.user.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.common.utils.TreeUtils;
import com.yd.user.api.dto.ApiSysDeptAddAndEditReqCheckDto;
import com.yd.user.api.service.ApiRelTenantDeptService;
import com.yd.user.api.service.ApiSysDeptService;
import com.yd.user.feign.request.sysdept.*;
import com.yd.user.feign.response.sysdept.ApiSysDeptCompanyPageResponse;
import com.yd.user.feign.response.sysdept.ApiSysDeptDetailResponse;
import com.yd.user.feign.response.sysdept.ApiSysDeptPageResponse;
import com.yd.user.feign.response.sysdept.ApiSysDeptTreeListResponse;
import com.yd.user.service.dto.RelTenantProjectDto;
import com.yd.user.service.dto.SysDeptDto;
import com.yd.user.service.model.*;
import com.yd.user.service.service.IRelTenantProjectService;
import com.yd.user.service.service.ISysDeptService;
import com.yd.user.service.service.ISysProjectService;
import com.yd.user.service.service.ISysTenantService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
public class ApiSysDeptServiceImpl implements ApiSysDeptService {

    @Autowired
    private IRelTenantProjectService iRelTenantProjectService;

    @Autowired
    private ISysDeptService iSysDeptService;

    @Autowired
    private ISysTenantService iSysTenantService;

    @Autowired
    private ISysProjectService iSysProjectService;

    @Autowired
    private ApiRelTenantDeptService apiRelTenantDeptService;

//    @Autowired
//    private ApiRelProjectDeptService apiRelProjectDeptService;

    /**
     * 树形部门列表查询
     * @param request
     * @return
     */
    @Override
    public Result<List<ApiSysDeptTreeListResponse>> treeList(ApiSysDeptTreeListRequest request) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的角色类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;
        //当前所属的租户id（当前页面右上角显示的租户） 超级管理员不传值，非超级管理员根据传值来
        if (!isAdmin && StringUtils.isBlank(request.getLoginTenantBizId())) {
            //非超级管理员并且LoginTenantBizId为空，提示必须传值LoginTenantBizId
            throw new BusinessException("非超级管理员，当前登录用户所属的租户id（当前页面右上角显示的租户）必须传值");
        }
        String loginTenantBizId = isAdmin ? "" : request.getLoginTenantBizId();

        List<ApiSysDeptTreeListResponse> apiSysDeptTreeListResponses = new ArrayList<>();
        List<ApiSysDeptTreeListResponse> treeMenu = new ArrayList<>();
        List<RelTenantProject> relTenantProjectList = new ArrayList<>();
        List<String> projectBizIdList = new ArrayList<>();

        if (!isAdmin) {
            //非超级管理员查询当前租户绑定的项目id列表，用于查询部门表绑定所属的部门
            relTenantProjectList = iRelTenantProjectService.queryList(RelTenantProjectDto.builder()
                    .tenantBizId(loginTenantBizId)
                    .build());
            if (!CollectionUtils.isEmpty(relTenantProjectList)) {
                projectBizIdList = relTenantProjectList.stream().map(RelTenantProject::getProjectBizId).collect(Collectors.toList());
            }
        }


        //条件搜索构建部门树的数据
        apiSysDeptTreeListResponses = iSysDeptService.treeList(loginTenantBizId,
                isAdmin,
                request.getDeptName(),
                projectBizIdList);

        if (!CollectionUtils.isEmpty(apiSysDeptTreeListResponses)) {
            //构建部门树形结构
            treeMenu = TreeUtils.buildTree(
                    apiSysDeptTreeListResponses,
                    ApiSysDeptTreeListResponse::getDeptBizId,         // 获取节点ID
                    ApiSysDeptTreeListResponse::getParentBizId,   // 获取父节点ID
                    (parent, children) -> parent.setChildren(children), // 设置子节点
                    "0"                   // 根节点的父ID值
            );
        }

        return Result.success(treeMenu);
    }

    /**
     * 部门分页查询
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiSysDeptPageResponse>> page(ApiSysDeptPageRequest request) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的角色类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;
        //当前所属的租户id（当前页面右上角显示的租户） 超级管理员不传值，非超级管理员根据传值来
        if (!isAdmin && StringUtils.isBlank(request.getLoginTenantBizId())) {
            //非超级管理员并且LoginTenantBizId为空，提示必须传值LoginTenantBizId
            throw new BusinessException("非超级管理员，当前登录用户所属的租户id（当前页面右上角显示的租户）必须传值");
        }
        String loginTenantBizId = isAdmin ? "" : request.getLoginTenantBizId();

        List<String> projectBizIdList = new ArrayList<>();
        Page<ApiSysDeptPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());

        if (!isAdmin) {
            //非超级管理员，查询当前租户下绑定的项目列表id
            List<RelTenantProject> relTenantProjectList = iRelTenantProjectService.queryList(RelTenantProjectDto.builder()
                    .tenantBizId(loginTenantBizId)
                    .build());
            if (CollectionUtils.isEmpty(relTenantProjectList)) {
                projectBizIdList = relTenantProjectList.stream().map(RelTenantProject::getProjectBizId).collect(Collectors.toList());
            }
        }

        IPage<ApiSysDeptPageResponse> iPage = iSysDeptService.page(page, request,isAdmin,projectBizIdList,loginTenantBizId);
        return Result.success(iPage);
    }

    /**
     * 公司分页查询（公司类型）
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiSysDeptCompanyPageResponse>> companyPage(ApiSysDeptCompanyPageRequest request) {
        Page<ApiSysDeptCompanyPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiSysDeptCompanyPageResponse> iPage = iSysDeptService.companyPage(page, request);
        return Result.success(iPage);
    }

    /**
     * 获取部门详情
     * @param deptBizId 部门唯一标识(业务ID)
     * @return
     */
    @Override
    public Result<ApiSysDeptDetailResponse> detail(String deptBizId) {
        Result<SysDept> result = checkSysDeptIsExist(deptBizId);
        SysDept sysDept = result.getData();

        ApiSysDeptDetailResponse response = new ApiSysDeptDetailResponse();
        BeanUtils.copyProperties(sysDept,response);

        if (CommonEnum.SCOPE_SYS.getCode().equals(response.getScope().toString())) {
            //系统级
            response.setTenantBizId("");
            response.setTenantName("");
            response.setProjectBizId("");
            response.setProjectName("");
        }else if (CommonEnum.SCOPE_TENANT.getCode().equals(response.getScope().toString())) {
            //租户级
            SysTenant sysTenant = iSysTenantService.queryOne(response.getTenantBizId());
            if (!Objects.isNull(sysTenant)) {
                response.setTenantName(sysTenant.getTenantName());
            }
            response.setProjectBizId("");
            response.setProjectName("");
        }else if (CommonEnum.SCOPE_PROJECT.getCode().equals(response.getScope().toString())) {
            //项目级
            SysProject sysProject = iSysProjectService.queryOne(response.getProjectBizId());
            if (!Objects.isNull(sysProject)) {
                response.setProjectName(sysProject.getProjectName());
            }
            response.setTenantBizId("");
            response.setTenantName("");
        }
        return Result.success(response);
    }

    /**
     * 添加部门
     * @param request
     * @return
     */
    @Override
    public Result add(ApiSysDeptAddRequest request) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的角色类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;

        ApiSysDeptAddAndEditReqCheckDto dto = new ApiSysDeptAddAndEditReqCheckDto();
        BeanUtils.copyProperties(request,dto);
        dto.setIsAdmin(isAdmin);
        dto.setOprSource(CommonEnum.OPR_SOURCE_ADD.getCode());
        //部门添加和编辑校验入参的合理性（非查询库校验）
        addAndEditReqCheck(dto);

        //校验部门名称唯一性
        List<SysDept> list = iSysDeptService.queryList(SysDeptDto.builder()
                .deptName(request.getDeptName())
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException(ResultCode.DEPT_NAME_EXISTS.getCode(),ResultCode.DEPT_NAME_EXISTS.getMessage());
        }

        SysDept sysDept = new SysDept();
        BeanUtils.copyProperties(request,sysDept);
        sysDept.setDeptBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_DEPT.getCode()));
        //添加和编辑保存设置参数公共方法
        sysDept = addAndEditSaveSetCommon(sysDept);
        iSysDeptService.saveOrUpdate(sysDept);

        //新增或者编辑-更新租户部门关系、更新项目部门关系公共方法
        addAndEditByRelCommon(request.getTenantBizId(),
                request.getProjectBizId(),
                sysDept.getDeptBizId(),
                request.getScope(),
                CommonEnum.OPR_SOURCE_ADD.getCode());

        return Result.success();
    }

    /**
     * 编辑部门
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiSysDeptEditRequest request) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的角色类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;

        ApiSysDeptAddAndEditReqCheckDto dto = new ApiSysDeptAddAndEditReqCheckDto();
        BeanUtils.copyProperties(request,dto);
        dto.setIsAdmin(isAdmin);
        dto.setOprSource(CommonEnum.OPR_SOURCE_EDIT.getCode());
        //部门添加和编辑校验入参的合理性（非查询库校验）
        addAndEditReqCheck(dto);

        //校验部门是否存在
        Result<SysDept> result = checkSysDeptIsExist(request.getDeptBizId());
        SysDept sysDept = result.getData();

        //校验部门名称唯一性
        List<SysDept> list = iSysDeptService.queryList(SysDeptDto.builder()
                .deptName(request.getDeptName())
                .deptBizId(request.getDeptBizId())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException(ResultCode.DEPT_NAME_EXISTS.getCode(),ResultCode.DEPT_NAME_EXISTS.getMessage());
        }

        //TODO 换绑逻辑（作用域换绑：提示校验需要先把什么解绑再来换绑）

        BeanUtils.copyProperties(request,sysDept);
        //添加和编辑保存设置参数公共方法
        sysDept = addAndEditSaveSetCommon(sysDept);
        iSysDeptService.saveOrUpdate(sysDept);

        //新增或者编辑-更新租户部门关系、更新项目部门关系公共方法 TODO
//        addAndEditByRelCommon(request.getTenantBizId(),
//                request.getProjectBizId(),
//                sysDept.getDeptBizId(),
//                request.getScope(),
//                CommonEnum.OPR_SOURCE_EDIT.getCode());

        return Result.success();
    }

    /**
     * 删除部门（逻辑删除）
     * @param deptBizId 部门唯一标识(业务ID)
     * @return
     */
    @Override
    public Result del(String deptBizId) {
        return null;
    }

    /**
     * 编辑部门状态
     * @param deptBizId 部门唯一标识(业务ID)
     * @param status 状态(0:禁用 1:启用)
     * @return
     */
    @Override
    public Result editStatus(String deptBizId, Integer status) {
        return null;
    }

    /**
     * 新增或者编辑-更新租户部门关系、更新项目部门关系公共方法
     * @param tenantBizId 租户id
     * @param projectBizId 项目id
     * @param deptBizId 部门id
     * @param scope 作用域
     * @param oprSource 操作来源 1-添加 2-编辑
     */
    public void addAndEditByRelCommon(String tenantBizId,
                                      String projectBizId,
                                      String deptBizId,
                                      Integer scope,
                                      String oprSource) {
        if (CommonEnum.OPR_SOURCE_ADD.getCode().equals(oprSource)) {
            //添加来源
            if (CommonEnum.SCOPE_TENANT.getCode().equals(scope.toString())) {
                //作用域租户级-新增租户部门关系
                apiRelTenantDeptService.addSingleRel(tenantBizId,deptBizId);
            }
            if (CommonEnum.SCOPE_PROJECT.getCode().equals(scope.toString())) {
                //作用域项目级-新增项目部门关系
//                apiRelProjectDeptService.addSingleRel(projectBizId,deptBizId);
            }
        }else if (CommonEnum.OPR_SOURCE_EDIT.getCode().equals(oprSource)) {
            //编辑来源 TODO
        }
    }
    
    /**
     * 添加和编辑保存设置参数公共方法
     * @param sysDept
     * @return
     */
    public SysDept addAndEditSaveSetCommon(SysDept sysDept) {
        if (CommonEnum.SCOPE_SYS.getCode().equals(sysDept.getScope().toString())) {
            //系统级
            sysDept.setTenantBizId("");
            sysDept.setProjectBizId("");
        }else if (CommonEnum.SCOPE_TENANT.getCode().equals(sysDept.getScope().toString())) {
            //租户级
            sysDept.setProjectBizId("");
        }else if (CommonEnum.SCOPE_PROJECT.getCode().equals(sysDept.getScope().toString())) {
            //项目级
            sysDept.setTenantBizId("");
        }
        return sysDept;
    }

    /**
     * 部门添加和编辑校验入参的合理性（非查询库校验）
     * @param dto
     */
    public void addAndEditReqCheck(ApiSysDeptAddAndEditReqCheckDto dto) {
        if (dto.getParentBizId().equals(dto.getDeptBizId())) {
            //父级部门不能为当前部门
            throw new BusinessException("父级部门不能为当前部门！!");
        }
        if (!dto.getIsAdmin() && CommonEnum.SCOPE_SYS.getCode().equals(dto.getScope().toString())) {
            //非超级管理员不能选择系统级作用域
            throw new BusinessException("非超级管理员不能选择系统级作用域！");
        }
        //校验作用域scope字段，如果为作用域租户级所属租户id必须传值
        if(CommonEnum.SCOPE_TENANT.getCode().equals(dto.getScope().toString())
                && StringUtils.isBlank(dto.getTenantBizId())) {
            throw new BusinessException("租户级作用域，所属租户id不能为空");
        }
        //校验作用域scope字段，如果为作用域项目级所属项目id必须传值
        if(CommonEnum.SCOPE_PROJECT.getCode().equals(dto.getScope().toString())
                && StringUtils.isBlank(dto.getProjectBizId())) {
            throw new BusinessException("项目级作用域，所属项目id不能为空");
        }
    }
    
    /**
     * 校验部门是否存在
     * @param deptBizId
     * @return
     */
    public Result<SysDept> checkSysDeptIsExist(String deptBizId) {
        SysDept sysDept = iSysDeptService.queryOne(deptBizId);
        if (Objects.isNull(sysDept)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(sysDept);
    }

}
